#!/bin/bash
# version: 1.12
# (c) Kristian Peters 2007-2020
# released under the terms of GPL
#
# changes: 1.12 - added support for macOS 10.15
#          1.11 - added support for latest macOS
#          1.10 - added support for OS X 10.10 Yosemite
#          1.9 - added support for Mac OS X 10.9
#          1.8 - added support for Mac OS X 10.8
#          1.7 - added removal of Icon^M, better matching of extended attributes
#          1.6 - added remove_acls
#          1.5 - updates for extended attributes for Mac OS X 10.7
#          1.4 - removing hidden extended attributes on hfs+ volumes
#          1.3 - files named "._[*]" are removed if "[*]" exists in the
#                same directory
#          1.2 - added .Trashes and ._.DS_Store from OSX
#          1.1 - added .trash-<username>
#          1.0 - initial release
#
# contact: "Kristian Peters" <kristian.peters@korseby.net>

IFS=$'\n'
USERNAME="$(whoami)"



function remove_ds_store() {
	# remove .DS_Store files (generated by the macOS Finder)
	for i in $(find . -name ".DS_Store" -print); do
		echo "Deleting \"${i}\"..."
		rm -f "${i}"
	done
}



function remove_icon() {
	# remove hidden Icon^M
	# BEWARE: line breaks may be wrong with editor
	#         use vi and <ctrl v> <ctrl m> for matching the ^M
	for i in $(find . -name "Icon
" -print); do
		echo "Moving \"$(echo $i | sed -e "s/
//")^M\" to Trash..."
		mv -f "${i}" "${HOME}/.Trash/"
	done
}



function remove_trashes() {
	# remove .Trashes files (generated by macOS Finder)
	for i in $(find . -name ".Trashes" -print); do
		echo "Moving \"${i}\" to Trash..."
		mv -f "${i}" "${HOME}/.Trash/"
	done

	# remove .Trash-<username> directories (generated by Nautilus from Gnome)
	for i in $(find . -name ".Trash-${USERNAME}" -print); do
		echo "Moving contents of \"${i}\" to Tash..."
		mv -f "${i}/*" "${HOME}/.Trash/"
		rmdir "${i}"
	done
}



function remove_acls() {
	# remove acl via chmod
	alias ls='ls -1l'
	for i in $(find . -print); do
		list="$(ls -1l "$i" | cut -c 11)"
		if [ "$list" == "+" ] ; then
			echo "Removing ACL from \"$i\"..."
			chmod -N "$i"
		fi
	done
}



function remove_extended_attributes_leos() {
	# remove ._* files (generated by Mac OSX's Finder) under the condition
	# that a file named * (without the ._ be the first letters) already exists
	for i in $(find . -name "._*" -print); do
		if [ -f "$(echo ${i} | sed -e "s/\/\._/\//")" ] ; then
			echo "Moving \"${i}\" to Trash..."
			mv -f "${i}" "${HOME}/.Trash/"
		fi
	done

	# remove hidden extended attributes via xattr
	alias ls='ls -1l'
	for i in $(find . -print); do
		list="$(ls -1l "$i" | cut -c 11)"
		if [ "$list" == "@" ] ; then
			attr="$(xattr -l "$i" | grep :$ | grep -v \  | sed -e "s/:$//")"
			if [ "$attr" != "" ] ; then
				for j in $attr; do
					echo "Removing \"$j\" from \"$i\"..."
					xattr -d "$j" "$i"
				done
				xattr -c "$i"
			fi
		fi
	done
}



function remove_extended_attributes_latest() {
	# remove ._* files (generated by the macOS Finder) under the condition
	# that a file named * (without the ._ be the first letters) already exists
	for i in $(find . -name "._*" -print); do
		if [ -f "$(echo ${i} | sed -e "s/\/\._/\//")" ] ; then
			echo "Moving \"${i}\" to Trash..."
			mv -f "${i}" "${HOME}/.Trash/"
		fi
	done


	# remove hidden extended attributes via xattr
	alias ls='ls -1l'
	for i in $(find . -print); do
		attr="$(xattr -l "$i" | grep -v ^00 | sed -e "s/[^\:]+$.*//" | sed -e "s/\:$//" | sed -e "s/\: .*//")"
		if [ "$attr" != "" ] ; then
			for j in $attr; do
				echo "Removing \"$j\" from \"$i\"..."
				xattr -d "$j" "$i"
			done
		fi
	done
}



# remove_ds_store
remove_ds_store

# remove_icon
remove_icon

# remove_trashes
remove_trashes

# remove_acls
remove_acls

# remove_extended_attributes
if [ -x "/usr/bin/sw_vers" ] ; then
	OS_VERSION="$(/usr/bin/sw_vers | grep ProductVersion | sed -e "s/.*10\.//" | sed -e "s/\..*//")"
	if [ "$OS_VERSION" -gt "7" ]; then
		remove_extended_attributes_latest
	elif [ "$OS_VERSION" == "6" ] || [ "$OS_VERSION" == "5" ] ; then
		remove_extended_attributes_leos
	else
		echo "Warning: Unsupported macOS Version."
		remove_extended_attributes_latest
	fi
else
	echo "Warning: No macOS detected."
	remove_extended_attributes_latest
fi
